<?php
/*
Plugin Name: Post/Page Specific JS & CSS
Description: Add JavaScript and CSS to individual posts/pages
Version: 2.0
*/

// Add custom field to post editor
add_action('add_meta_boxes', function() {
	// JavaScript box
	add_meta_box('post_js_box', 'Post JavaScript', 'post_js_meta_box', 'post', 'normal', 'high');
	add_meta_box('post_js_box', 'Page JavaScript', 'post_js_meta_box', 'page', 'normal', 'high');
	
	// CSS box
	add_meta_box('post_css_box', 'Post CSS', 'post_css_meta_box', 'post', 'normal', 'high');
	add_meta_box('post_css_box', 'Page CSS', 'post_css_meta_box', 'page', 'normal', 'high');
});

// JavaScript meta box
function post_js_meta_box($post) {
	wp_nonce_field('post_js_nonce', 'post_js_nonce_field');
	$value = get_post_meta($post->ID, '_post_javascript', true);
	echo '<textarea name="post_javascript" style="width:100%;height:150px;font-family:monospace;" placeholder="// Enter JavaScript code here (without &lt;script&gt; tags)">' . esc_textarea($value) . '</textarea>';
	echo '<p><small>Enter JavaScript without &lt;script&gt; tags. jQuery is available if your theme loads it.</small></p>';
	
	// Show jQuery example
	echo '<details style="margin-top:10px;background:#f5f5f5;padding:10px;">';
	echo '<summary style="cursor:pointer;font-weight:bold;">jQuery Example</summary>';
	echo '<pre style="background:#fff;padding:10px;margin-top:5px;">';
	echo "jQuery(document).ready(function($) {\n";
	echo "    // Your jQuery code here\n";
	echo "    $('.post-title').css('color', 'red');\n";
	echo "});";
	echo '</pre>';
	echo '</details>';
}

// CSS meta box
function post_css_meta_box($post) {
	wp_nonce_field('post_css_nonce', 'post_css_nonce_field');
	$value = get_post_meta($post->ID, '_post_css', true);
	echo '<textarea name="post_css" style="width:100%;height:150px;font-family:monospace;" placeholder="/* Enter CSS code here (without &lt;style&gt; tags) */">' . esc_textarea($value) . '</textarea>';
	echo '<p><small>Enter CSS without &lt;style&gt; tags. Use !important if needed to override theme styles.</small></p>';
	
	// Show CSS examples
	echo '<details style="margin-top:10px;background:#f5f5f5;padding:10px;">';
	echo '<summary style="cursor:pointer;font-weight:bold;">CSS Examples</summary>';
	echo '<pre style="background:#fff;padding:10px;margin-top:5px;">';
	echo "/* Change post title color */\n";
	echo ".entry-title {\n";
	echo "    color: #ff0000 !important;\n";
	echo "}\n\n";
	echo "/* Add border to images in this post */\n";
	echo ".post-" . $post->ID . " img {\n";
	echo "    border: 2px solid #ccc;\n";
	echo "    padding: 5px;\n";
	echo "}\n\n";
	echo "/* Make paragraphs larger */\n";
	echo ".post-" . $post->ID . " p {\n";
	echo "    font-size: 18px;\n";
	echo "    line-height: 1.6;\n";
	echo "}";
	echo '</pre>';
	echo '</details>';
}

// Save custom fields
add_action('save_post', function($post_id) {
	// Save JavaScript
	if (isset($_POST['post_js_nonce_field']) && wp_verify_nonce($_POST['post_js_nonce_field'], 'post_js_nonce')) {
		if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
		if (!current_user_can('edit_post', $post_id)) return;
		
		if (isset($_POST['post_javascript'])) {
			update_post_meta($post_id, '_post_javascript', $_POST['post_javascript']);
		} else {
			delete_post_meta($post_id, '_post_javascript');
		}
	}
	
	// Save CSS
	if (isset($_POST['post_css_nonce_field']) && wp_verify_nonce($_POST['post_css_nonce_field'], 'post_css_nonce')) {
		if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
		if (!current_user_can('edit_post', $post_id)) return;
		
		if (isset($_POST['post_css'])) {
			update_post_meta($post_id, '_post_css', $_POST['post_css']);
		} else {
			delete_post_meta($post_id, '_post_css');
		}
	}
});

// Output JavaScript in footer
add_action('wp_footer', function() {
	if (is_singular()) {
		global $post;
		
		// Output JavaScript
		$js = get_post_meta($post->ID, '_post_javascript', true);
		if (!empty($js)) {
			echo "\n<!-- Post-Specific JavaScript -->\n";
			echo "<script>\n";
			echo "(function() {\n";
			echo "try {\n";
			echo stripslashes(trim($js)) . "\n";
			echo "} catch(e) {\n";
			echo "console.error('Post JS Error on post #" . $post->ID . ":', e);\n";
			echo "}\n";
			echo "})();\n";
			echo "</script>\n";
		}
		
		// Output CSS in footer (as fallback)
		$css = get_post_meta($post->ID, '_post_css', true);
		if (!empty($css)) {
			echo "\n<!-- Post-Specific CSS (in footer) -->\n";
			echo "<style>\n";
			echo stripslashes(trim($css)) . "\n";
			echo "</style>\n";
		}
	}
});

// Output CSS in header (primary location)
add_action('wp_head', function() {
	if (is_singular()) {
		global $post;
		$css = get_post_meta($post->ID, '_post_css', true);
		if (!empty($css)) {
			echo "\n<!-- Post-Specific CSS -->\n";
			echo "<style>\n";
			echo stripslashes(trim($css)) . "\n";
			echo "</style>\n";
		}
	}
});

// Add admin styles for better appearance
add_action('admin_head', function() {
	echo '<style>
	.post-js-box textarea,
	.post-css-box textarea {
		background: #f8f8f8;
		border: 1px solid #ddd;
		border-radius: 4px;
		padding: 10px;
	}
	.post-js-box textarea:focus,
	.post-css-box textarea:focus {
		background: #fff;
		border-color: #007cba;
		box-shadow: 0 0 0 1px #007cba;
	}
	.post-js-box details,
	.post-css-box details {
		border: 1px solid #ddd;
		border-radius: 4px;
	}
	</style>';
});